function [fig, ax] = engMapPlot(eng, contour_type, yax, normalized)
arguments
    eng struct
    contour_type {mustBeTextScalar, mustBeMember(contour_type, ["fc", "NOx", "HC", "PM", "CO", "bsfc", "bsNOx", "bsHC", "bsPM", "bsCO"])}  = "fc"
    yax string {mustBeMember(yax, ["power", "torque"])} = "torque"
    normalized = false
end
%engMapPlot
% Plot the engine fuel consumption, brake specific fuel consumption or
% efficiency map.
%
% Input arguments
% ---------------
% eng : struct
%   engine data structure.
% contour_type : string, optional
%   Specify: 'fc' to plot the fuel flow rate map;
%            'bsfc' (default) for brake specific fuel consumption;
%            'eff' for fuel conversion efficiency.
% yax : string, optional
%   Specify: 'torque'(default) to plot torque vs speed;
%            'power' to plot power vs speed.
% normalized : logical, optional
%   Specify: 'true' to normalize the axes and contour plot;
%            'false' (default) otherwise.
%
% Outputs
% ---------------
% fig : Figure
%   Figure handle of the plot.
% ax : Axes
%   Axes handle of the plot.

%%
if strcmp(contour_type, 'fc')
    contour_type = 'fuelFlwRate';
end
if isempty(yax)
    yax = "torque";
end

% Create contour plot data
spdBrk = linspace(eng.idleSpd, eng.maxSpd, 200);
switch yax
    case "torque"
        trqBrk = linspace(eng.fuelFlwRate.GridVectors{2}(1), eng.fuelFlwRate.GridVectors{2}(end), 220);
        [spdMesh, trqMesh] = ndgrid(spdBrk, trqBrk);
        yMesh = trqMesh;
    case "power"
        pwrBrk = linspace(0, eng.ratedPwr*1e3, 220);
        [spdMesh, pwrMesh] = ndgrid(spdBrk, pwrBrk);
        yMesh = pwrMesh .* 1e-3; % kW
        trqMesh = pwrMesh ./ spdMesh; % W
end

contour_data = eng.(contour_type)(spdMesh, trqMesh); % kg/s or K

% Remove contour plot data above WOT curve
maxTrq = eng.maxTrq(spdMesh);
contour_data(trqMesh>maxTrq) = nan;

% Levels
switch contour_type
    case {'fuelFlwRate', 'NOx', 'HC', 'PM', 'CO'}
        contour_data = contour_data * 1e3; % g/s
        levels = linspace(min(contour_data(:)), max(contour_data(:)), 20);
        levels = round(levels, 2);
    case {'bsfc', 'bsNOx', 'bsHC', 'bsPM', 'bsCO'}
        contour_data(contour_data<0) = 0;
        contour_data(isinf(contour_data)) = nan;
        levels = min(contour_data(:)) + (max(contour_data(:)) - min(contour_data(:))) .* linspace(0, 1, 20).^3;
        levels = round(levels, 2);
end

% Labels and overrides
switch contour_type
    case 'fuelFlwRate'
        contour_legend_string = 'Fuel consumption, g/s';
    case 'bsfc'
        levels = round(levels, 0);
        levels = [210:1:215, 217, 220, 225, 230:10:260, 275:25:300, 400, 600];
        levels = [180:5:215, 220:10:260, 275:25:300, 400, 600];
        contour_legend_string = 'bsfc, g/kWh';
    case 'eff'
        levels = [0.5, 0.1, 0.2, 0.25, 0.28:0.02:0.4];
        contour_legend_string = 'Efficiency';
    case 'NOx'
        contour_legend_string = 'NOx, g/s';
    case 'bsNOx'
        levels = [3.8 3.9 4 4.1 4.2 4.5 5 5.5 6 6.5 7 9 12 14 16 18 20];
        contour_legend_string = 'NOx, g/kWh';
    case 'HC'
        levels = [0.0006:0.0006:0.0042 0.0084];
        contour_legend_string = 'HC, g/s';
    case 'bsHC'
        levels = [0.025 0.04 0.06 0.09 0.12 0.18 0.36 0.72 1.44 5.76];
        contour_legend_string = 'HC, g/kWh';
    case 'PM'
        contour_legend_string = 'PM, g/s';
    case 'bsPM'
        contour_legend_string = 'PM, g/kWh';
    case 'CO'
        contour_legend_string = 'CO, g/s';
    case 'bsCO'
        levels = [0.09 0.18 0.36 0.72 1.44 2.88 5.32];
        contour_legend_string = 'CO, g/kWh';
end
colorScale = [levels(1) levels(end)];


%% Plot engine map
fig = figure;
ax = axes;
hold on
grid on

spdBrkRPM = spdBrk .* 30/pi;
spdMeshRPM = spdMesh .* 30/pi;

% draw WOT
maxTrq = eng.maxTrq(spdBrk);
switch yax
    case "torque"
        yMax = maxTrq;
    case "power"
        maxPwr = maxTrq .* spdBrk; % W
        yMax = maxPwr * 1e-3; % kW
end
plot(ax, spdBrkRPM, yMax, 'k', 'LineWidth', 2, 'HandleVisibility', 'off')

% null torque line
plot(ax, [min(spdBrkRPM) max(spdBrkRPM)], [0, 0], 'k', 'LineWidth', 1, 'HandleVisibility', 'off')

% motoring curve
if isfield(eng, 'motTrq')
    minTrq = eng.motTrq(spdBrk);
else
    minTrq = zeros(size(spdBrk));
end
switch yax
    case "torque"
        yMin = minTrq;
    case "power"
        minPwr = minTrq .* spdBrk; % W
        yMin = minPwr * 1e-3; % kW
end
plot(spdBrkRPM, yMin, 'k', 'LineWidth', 2, 'HandleVisibility', 'off')

% contour plot
if normalized
    shwTxt = 'off';
else
    shwTxt = 'on';
end

[c, h] = contour(ax, spdMeshRPM, yMesh, contour_data, levels, 'ShowText', shwTxt, 'LabelSpacing', 1440);
if ~normalized
 clabel(c, h, 'LabelSpacing', 1440, 'color', 'k', 'FontWeight', 'bold')
end

caxis(colorScale)

% draw OOL
if isfield(eng, 'oolTrq')
    oolTrq = eng.oolTrq(spdBrk);
    plot(ax, spdBrkRPM, oolTrq, '--', 'Color', [0.5 0.5 0.5], 'LineWidth', 2)
end

%% Finalize figure
xlim([eng.idleSpd eng.maxSpd].*30/pi)
switch yax
    case "torque"
        ylim([min(minTrq) max(trqBrk)])
    case "power"
        ylim([0 max(yMax)])
end

ylims = ylim;
xlabel('Speed, RPM')

switch yax
    case "torque"
        fill([spdBrkRPM spdBrkRPM(end) spdBrkRPM(1)], [maxTrq ylims(2) ylims(2)], 0.9*ones(1,3))
        fill([spdBrkRPM spdBrkRPM(end) spdBrkRPM(1)], [minTrq ylims(1) ylims(1)], 0.9*ones(1,3))
        ylabel('Torque, Nm')
    case "power"
        fill([spdBrkRPM spdBrkRPM(end) spdBrkRPM(1)], [yMax ylims(2) ylims(2)], 0.9*ones(1,3))
        fill([spdBrkRPM spdBrkRPM(end) spdBrkRPM(1)], [yMin ylims(1) ylims(1)], 0.9*ones(1,3))
        ylabel('Power, kW')
end

if isfield(eng, 'oolTrq')
    legend(contour_legend_string, 'OOL')
else
    legend(contour_legend_string)
end

if normalized
    ax.YTickLabel{1} = '0';
    ax.YTickLabel(2:end-1) = {' '};
    ax.YTickLabel{end} = 'T_{max}';
    ax.XTickLabel{1} = '\omega_{idle}';
    ax.XTickLabel(2:end-1) = {' '};
    ax.XTickLabel{end} = '\omega_{max}';
end

end